/***********************************************************************/
/*  This file is part of the ARM Toolchain package                     */
/*  Copyright KEIL ELEKTRONIK GmbH 2003 - 2010                         */
/***********************************************************************/
/*                                                                     */
/*  FlashDev.C:  Flash Programming Functions adapted                   */
/*               for ES32F3xx Flash                                    */
/*                                                                     */
/***********************************************************************/
#include "FlashOS.H"        /* FlashOS Structures */
#include "FlashParams.h"    /* flash parameters specification */
#include "MCUParams.h"

qspi_dac_cfg_t g_dac;
qspi_handle_t hqspi = {0};
qspi_data_capture_cfg_t g_capture_cfg = {0};

/*
 *  Initialize QSPI for default configuration
 *    Return Value:   0 - OK,  1 - Failed
 */
int qspi_int(void)
{

    #ifdef OutputDebugInfo
    DebugPrintf("QSPI config as default, start...\n\r");
    #endif

    hqspi.perh         = QSPI;
    hqspi.init.clkdiv  = QSPI_DIV_4;
    hqspi.init.cpol    = QSPI_CPOL_L;
    hqspi.init.chpa    = QSPI_CPHA_1E;
    hqspi.init.chipsel = QSPI_CS_NSS0;
    hqspi.init.wrppin  = DISABLE;
    ald_qspi_init(&hqspi);

    g_dac.dtrprtcol  = DISABLE;
    g_dac.ahbdecoder = DISABLE;
    g_dac.xipimmed 	 = DISABLE;
    g_dac.xipnextrd  = DISABLE;
    g_dac.addrremap  = DISABLE;
    g_dac.dmaperh 	 = DISABLE;

    g_dac.wrinit.instxfer = QSPI_XFER_SINGLE;  
    g_dac.wrinit.addxfer = QSPI_XFER_SINGLE;
    g_dac.wrinit.datxfer = QSPI_XFER_SINGLE;
    g_dac.wrinit.autowel = ENABLE;
    g_dac.wrinit.wrcde   = 0x2;
    g_dac.wrinit.dcyles  = 0;

    g_dac.rdinit.instxfer = QSPI_XFER_SINGLE;   
    g_dac.rdinit.addxfer  = QSPI_XFER_SINGLE;
    g_dac.rdinit.datxfer  = QSPI_XFER_SINGLE;
    g_dac.rdinit.ddrbit   = DISABLE;
    g_dac.rdinit.modebit  = DISABLE;  
    g_dac.rdinit.mbitval  = 0x00;
    g_dac.rdinit.rdcde    = 0x03;  
    g_dac.rdinit.dcyles   = 0;

    config_flash_param(&hqspi);
    qspi_dac_config(&hqspi, &g_dac);

    #ifdef OutputDebugInfo
    DebugPrintf("Configure QSPI success\n\r");
    #endif	

    if(bsp_w25q128_reset() != OK)
        return (1);

    if(bsp_w25q128_disable_protect() != OK)
        return (1);

    return 0;
}


/*
 *  Initialize Flash Programming Functions
 *    Parameter:      adr:  Device Base Address
 *                    clk:  Clock Frequency (Hz)
 *                    fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
 *    Return Value:   0 - OK,  1 - Failed
 */
int Init(unsigned long adr, unsigned long clk, unsigned long fnc)
{

    /* Configure system clock: HRC 24MHz */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    SysTick->CTRL = 0x0UL;

    #ifdef OutputDebugInfo
    Debug_Output_Init();
    DebugPrintf("Init Start...\n\r");
    #endif
        
    /* HCLK2 for QSPI*/
    ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_HCLK2);
    /* Enable peripherals clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Init QSPI pins */
    qspi_pin_init();

    /* Config QSPI */
    qspi_int();

    return (0);
}

/*
 *  De-Initialize Flash Programming Functions
 *    Parameter:      fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
 *    Return Value:   0 - OK,  1 - Failed
 */
int UnInit(unsigned long fnc)
{
#ifdef OutputDebugInfo
    DebugPrintf("DeInit Start\n\r");
#endif

    ald_rmu_reset_periperal(RMU_PERH_GPIO);
    ald_rmu_reset_periperal(RMU_PERH_QSPI);

#ifdef OutputDebugInfo
    ald_rmu_reset_periperal(RMU_PERH_UART0);
    DebugPrintf("DeInit Done\n\r");
#endif

return (0);
}

/*
 *  Erase complete Flash Memory
 *    Return Value:   0 - OK,  1 - Failed
 */

int EraseChip(void) 
{
#ifdef OutputDebugInfo
    DebugPrintf("Chip Erase Start\n\r");
#endif

    if (0x00 != bsp_w25q128_erase_chip())
    {
#ifdef OutputDebugInfo
    DebugPrintf("Chip Erase Error\n\r");
#endif
        return (1);
    }
    else
    {
#ifdef OutputDebugInfo
    DebugPrintf("Chip Erase Done\n\r");
#endif
        return (0);
    }
}

/*
 *  Erase Sector in Flash Memory
 *    Parameter:      adr:  Sector Address
 *    Return Value:   0 - OK,  1 - Failed
 */
int EraseSector(unsigned long adr)
{
#ifdef OutputDebugInfo
    DebugPrintf("Sector Erase Start, address: 0x%x\n\r", adr);
#endif
    
    if (0x00 != bsp_w25q128_erase_block(adr))
    {
#ifdef OutputDebugInfo
    DebugPrintf("Sector Erase Done\n\r");
#endif
        return (1);
    }
    else
    {
        return (0);
    }
}

/*
 *  Program Page in Flash Memory
 *    Parameter:      adr:  Page Start Address
 *                    sz:   Data Length
 *                    buf:  Page Data
 *    Return Value:   0 - OK,  1 - Failed
 */
int ProgramPage(unsigned long adr, unsigned long sz, unsigned char *buf)
{
    uint32_t i;

#ifdef OutputDebugInfo
    DebugPrintf("Page Program Start, start address: 0x%X, data length : 0x%x\n\r", adr, sz);
#endif

    for (i = 0; i < sz / 4; i++)
    {
        *((volatile uint32_t *)adr + i) = *((volatile uint32_t *)buf + i);
    }
#ifdef OutputDebugInfo
    DebugPrintf("Page Program Done\n\r");
#endif

    return 0;
}

 /*
  *  Verify Flash Contents
  *    Parameter:      adr:  Start Address
  *                    sz:   Size (in bytes)
  *                    buf:  Data
  *    Return Value:   (adr+sz) - OK, Failed Address
  */
unsigned long Verify(unsigned long adr, unsigned long sz, unsigned char *buf)
{
    uint32_t i;

#ifdef OutputDebugInfo
    DebugPrintf("Verify ...\n\r");
    DebugPrintf("Content length: %d\n\r", sz);
    DebugPrintf("QSPI->CR: 0x%X\n\r", QSPI->CR);
    DebugPrintf("QSPI->DRIR: 0x%X\n\r", QSPI->DRIR);
    DebugPrintf("QSPI->DWIR: 0x%X\n\r", QSPI->DWIR);
#endif

    for (i = 0; i < (sz / 4); i++)
    {
#ifdef OutputDebugInfo
        DebugPrintf("Content in Flash: 0x%X is 0x%X, and in hex file: 0x%X\n\r", adr + i, *((uint32_t *)(adr + i)), *((volatile uint32_t *)buf + i));
#endif
        if (*((volatile uint32_t *)adr + i) != *((volatile uint32_t *)buf + i))
            break;
    }

    i *= 4;

    return (adr + i);
}


int BlankCheck(unsigned long adr, unsigned long sz, unsigned char pat)
{
#ifdef OutputDebugInfo
    DebugPrintf("Blank Check Done\n\r");
#endif

    return 1;                                        /* Always Force Erase */
}
